# -------------------------------------------------------------------------------------------------
#                            OUTRUN: AMIGA EDITION -  ROM EXPORTER v0.4
#                                        REASSEMBLER (2025)
#
# This script converts the OutRun Arcade ROMs into a format suitable for Outrun: Amiga Edition.
# This original graphics and sound data are not included in this package due to copyright.
#
# Requirements:
# 1/ OutRun Arcade ROMs (Most revisions work and will be CRC32 verified by this tooling)
# 2/ A working python installation (tested with 3.12)
# 3/ Librosa and Numpy Libraries installed (used for audio conversion) as follows:
#    pip install librosa (tested with 0.10.2)
#    pip install numpy   (tested with 2.1.3)
#
# Instructions:
# 1/ Copy the Arcade ROMs to the ‘in’ subdirectory.
# 2/ Run this script: python amiga_rom_exporter.py
# 3/ Newly created files are generated in the 'gfx' and 'audio' subdirectories.
# 4/ Transfer these subdirectories onto the Amiga.
# -------------------------------------------------------------------------------------------------

import os
import zlib

# Audio Imports
import librosa
import numpy as np

# -------------------------------------------------------------------------------------------------
# FILE OPERATIONS
# -------------------------------------------------------------------------------------------------
def read_file(filename):
    if not os.path.exists(filename):
        print(f"Failed to open expected file: {filename}")
        exit()

    with open(filename, "rb") as f:
        data = f.read()
    return data


def save_to_file(input_bytes, filename):
    try:
        with open(filename, "wb") as fo:
            fo.write(input_bytes)
    except PermissionError:
        print(f"Failed to write to {filename}")
        exit()
    return


# Load and Interleave a System 16 ROM
def load_rom(filename, destination: bytearray, offset, length, crc32, interleave=4):
    rom = read_file(filename)
    if rom == 0:
        exit()

    found_crc32 = zlib.crc32(rom)
    if found_crc32 != crc32:
        print(f"{filename}: Incorrect CRC32 value. Expected: {hex(crc32)} | Found: {hex(found_crc32)}")
        exit()

    for i in range(length):
        destination[(i * interleave) + offset] = rom[i]
    return True


def write_long(data: bytearray, offset: int, v: int):
    data[offset + 3] = v & 0xFF
    data[offset + 2] = (v >> 8) & 0xFF
    data[offset + 1] = (v >> 16) & 0xFF
    data[offset + 0] = (v >> 24) & 0xFF


# -------------------------------------------------------------------------------------------------
# ROAD EXPORTER
# -------------------------------------------------------------------------------------------------
SRC_WIDTH = 512
SRC_HEIGHT = 256


def load_road():
    ROAD_LENGTH = 0x8000
    road_data = bytearray(ROAD_LENGTH)
    load_rom("in/opr-10185.11", road_data, 0x00000, ROAD_LENGTH, 0x22794426,1)
    output = bytearray(SRC_WIDTH * SRC_HEIGHT)

    for y in range(SRC_HEIGHT):
        src = ((y & 0xff) * 0x40 + (y >> 8) * 0x8000) % ROAD_LENGTH
        line = y * SRC_WIDTH
        for x in range(SRC_WIDTH):
            output[line + x]  = ((road_data[src + int(x / 8)] >> (~x & 7)) & 1) << 0
            output[line + x] |= ((road_data[src + int(x / 8 + 0x4000)] >> (~x & 7)) & 1) << 1

            # pre-mark road data in the "stripe" area with a high bit
            if 256 - 8 <= x < 256 and output[line + x] == 3:
                output[line + x] |= 4
    return output


def convert_road1(road_data: bytearray):
    output = bytearray(SRC_WIDTH * SRC_HEIGHT)
    output[:] = road_data
    for y in range(SRC_HEIGHT):
        line = y * SRC_WIDTH
        for x in range(SRC_WIDTH):
            if road_data[line + x] == 3:          # Ground Colour
                output[line + x] = 0
            elif road_data[line + x] == 7:        # Stripe Colour
                output[line + x] = 3
            elif road_data[line + x] == 0:        # Old Road Colour
                output[line + x] = 3
    return output


def convert_road2(road_data: bytearray):
    output = bytearray(SRC_WIDTH * SRC_HEIGHT)
    output[:] = road_data
    for i in range(len(road_data)):
        if road_data[i] > 0:
            output[i] += 3
    return output


# -------------------------------------------------------------------------------------------------
# TILE EXPORTER
# -------------------------------------------------------------------------------------------------

# 16 Colour Remap for certain tilemaps
# Format: Start Tile, End Tile (Inclusive), Source Pixel, Target Pixel
tile_pixel_remap = [
   [0x14C1, 0x14FD, 1, 9],      # Stage 3c: BG Layer. Map Cloud Pixel 1 -> 9 in FG palette Region.
   [0x14C1, 0x14FD, 2, 9],      # Stage 3c: BG Layer. Map Cloud Pixel 2 -> 9 in FG palette Region.
   [0x14C1, 0x14FD, 3, 10],     # Stage 3c: BG Layer. Map Cloud Pixel 3 -> 10 in FG palette Region.
   [0x14C1, 0x14FD, 4, 10],     # Stage 3c: BG Layer. Map Cloud Pixel 4 -> 10 in FG palette Region.
   [0x14C1, 0x14FD, 5, 7],      # Stage 3c: BG Layer. Map Cloud Pixel 5 -> 7 in BG palette Region.
   [0x14C1, 0x14FD, 6, 1],      # Stage 3c: BG Layer. Map Cloud Pixel 6 -> 1 in BG palette Region.
   [0xA81, 0xAF2, 1, 2],        # Stage 5b: FG Layer
   [0xA81, 0xAF2, 6, 4],        # Stage 5b: FG Layer
   [0x1F41, 0x1FAC, 1, 9],      # Stage 5b: BG Layer. Map BG Pixel 1 -> 9 in FG Palette Region.
   [0x1F41, 0x1FAC, 2, 14],     # Stage 5b: BG Layer. Map BG Pixel 2 -> 14 in FG Palette Region.
   [0x1F41, 0x1FAC, 3, 15]      # Stage 5b: BG Layer. Map BG Pixel 3 -> 15 in FG Palette Region.
]

# Convert Tile ROMs to one block of data
def load_tiles():
    tile_data = bytearray(0x30000)
    load_rom("in/opr-10268.99", tile_data,  0x00000, 0x08000, 0x95344b04, 1)
    load_rom("in/opr-10232.102", tile_data, 0x08000, 0x08000, 0x776ba1eb, 1)
    load_rom("in/opr-10267.100", tile_data, 0x10000, 0x08000, 0xa85bb823, 1)
    load_rom("in/opr-10231.103", tile_data, 0x18000, 0x08000, 0x8908bcbf, 1)
    load_rom("in/opr-10266.101", tile_data, 0x20000, 0x08000, 0x9f6f1a74, 1)
    load_rom("in/opr-10230.104", tile_data, 0x28000, 0x08000, 0x686f5e50, 1)
    return tile_data


# Convert Tile ROMs to MAME Style Chunky
def convert_tiles(tile_data: bytearray):
    TILES_LENGTH = 0x10000
    output = bytearray(TILES_LENGTH * 4)

    for i in range(TILES_LENGTH):
        p0 = tile_data[i]
        p1 = tile_data[i + 0x10000]
        p2 = tile_data[i + 0x20000]
        val = 0

        for ii in range(8):
            bit = 7 - ii
            pix = (((p0 >> bit) & 1) | (((p1 >> bit) << 1) & 2) | (((p2 >> bit) << 2) & 4))
            val = (val << 4) | pix

        output[(i * 4) + 0] = (val >> 24) & 0xff
        output[(i * 4) + 1] = (val >> 16) & 0xff
        output[(i * 4) + 2] = (val >> 8) & 0xff
        output[(i * 4) + 3] = (val >> 0) & 0xff
    return output


def remap_tile_pixels(tile_data: bytearray):
    for entry in tile_pixel_remap:
        start_tile_adr = (entry[0] * 4 * 8)
        end_tile_adr = (entry[1] * 4 * 8)
        src_pixel = entry[2]
        remap_pixel = entry[3]

        for i in range(start_tile_adr,end_tile_adr):
            p0 = tile_data[i] & 0xf
            p1 = (tile_data[i] >> 4) & 0xf
            if p0 == src_pixel:
                p0 = remap_pixel
            if p1 == src_pixel:
                p1 = remap_pixel
            tile_data[i] = ((p1 << 4) | p0)

    return tile_data

# -------------------------------------------------------------------------------------------------
# SPRITE EXPORTER
# -------------------------------------------------------------------------------------------------
# Man Frames To Remap
sprite_man_adr = [
    [0x03EE6, 0x04795, 0xa],    # Flip Frames
    [0x29410, 0x296E4, 0xa],    # Spin Frames
    [0x297DB, 0x2985B, 0xa],    # Normal Frames
    [0x2D847, 0x2D895, 0xa],    # Reaction 1
    [0x2D99E, 0x2D9C8, 0xa]     # Reaction 2
]

# Female Pixels To Remap
sprite_woman_adr = [
    [0x04795, 0x05100, 0xa],    # Flip Frames (End Address is wrong!)
    [0x296E4, 0x297DB, 0xa],    # Spin Frames
    [0x2985B, 0x298A3, 0xa],    # Normal Frames
    [0x2d7cd, 0x2D847, 0xa],    # Reaction 1
    [0x2D895, 0x2D99E, 0xa]     # Reaction 2
]

# Format: Start of block, End of block, palette to remap transparent colour to
sprite_transparency_map = [
    [0x10000, 0x10694, 0xa],      # Shadow Sprite (For Scenery)
    [0x15817, 0x158F1, 0xa],      # Shadow Sprite (For Player Traffic)
    [0x35A5E, 0x35AA0, 0xa],      # Shadow Sprite (For Traffic)
    #[0x138EC, 0x15816, 0xd],      # Grass Sprite
    [0x00000, 0x00F99, 0xa],      # Ferrari Sprite
    [0x02489, 0x03103, 0xa],      # Ferrari Spin Sprite
    [0x06171, 0x064E0, 0xa],      # Ferrari End Sprite
    [0x2B14E, 0x2D7CD, 0xa],      # Ferrari Flip Sprite
    [0x1ECDB, 0x1FAF7, 0xa],      # Stage 1: Start Adverts Left
    [0x206BA, 0x20D7A, 0xa],      # Stage 1: Start Adverts Right
    [0x39715, 0x399AB, 0xe],      # Stage 1: Start Guy 3 - 5
    [0x286A1, 0x289E9, 0xa],      # Stage 1: Windsurfers
    [0x0BA22, 0x0C13D, 0xd],      # Stage 1: Bush 3 & 4
    [0x185D2, 0x1868f, 0x7],      # Stage 1: Oil Sign
    #[0x24AC8, 0x25F35, 0xe],      # Flag Man 1
    #[0x278F3, 0x27E4A, 0xe],      # Flag Man 2
    sprite_man_adr[0], sprite_man_adr[1], sprite_man_adr[2], sprite_man_adr[3], sprite_man_adr[4],
    sprite_woman_adr[0], sprite_woman_adr[1], sprite_woman_adr[2], sprite_woman_adr[3], sprite_woman_adr[4],
    [0x10948, 0x12A38, 0xa],      # Water Sprite
    [0x158F0, 0x15B1F, 0x1],      # Smoke Sprite
    [0x28D4F, 0x2924E, 0x1],      # Tyre Spray
    [0x35173, 0x35A5D, 0x0],      # Water Spray (Remap Transparency 0)
    [0x06C32, 0x07f00, 0xa],      # Map Pieces #1
    [0x26E16, 0x2720C, 0xa],      # Map Pieces #2
    [0x342B1, 0x342C5, 0xa],      # Radio Sprite
    [0x298F7, 0x2A3A9, 0xa],      # Logo: Oval Background
    [0x2A70E, 0x2AD7D, 0xa],      # Logo: Palm
    [0x2AF35, 0x2B118, 0xa]       # Logo: Car
]


# Pixel Mapping
sprite_man_remap   = [0, 12, 13, 3, 4, 5, 9, 11, 6, 7, 0, 3, 14, 10, 10, 15]
sprite_woman_remap = [0, 1, 2, 3, 4, 5, 4, 8, 8, 9, 0, 11, 11, 13, 11, 15]


# Convert 4bpp System 16 Sprite Roms to Chunky Format
def load_sprites():
    sprite_data = bytearray(0x100000)
    load_rom("in/mpr-10371.9",  sprite_data, 0x000000, 0x20000, 0x7cc86208)
    load_rom("in/mpr-10373.10", sprite_data, 0x000001, 0x20000, 0xb0d26ac9)
    load_rom("in/mpr-10375.11", sprite_data, 0x000002, 0x20000, 0x59b60bd7)
    load_rom("in/mpr-10377.12", sprite_data, 0x000003, 0x20000, 0x17a1b04a)
    load_rom("in/mpr-10372.13", sprite_data, 0x080000, 0x20000, 0xb557078c)
    load_rom("in/mpr-10374.14", sprite_data, 0x080001, 0x20000, 0x8051e517)
    load_rom("in/mpr-10376.15", sprite_data, 0x080002, 0x20000, 0xf3b8f318)
    load_rom("in/mpr-10378.16", sprite_data, 0x080003, 0x20000, 0xa1062984)
    return sprite_data


def convert_sprites(sprite_data: bytearray):
    length = len(sprite_data)
    output = bytearray(length)

    for i in range(int(length / 4)):
        d3 = sprite_data[(i * 4) + 0]
        d2 = sprite_data[(i * 4) + 1]
        d1 = sprite_data[(i * 4) + 2]
        d0 = sprite_data[(i * 4) + 3]
        write_long(output, i*4, (d0 << 24) | (d1 << 16) | (d2 << 8) | d3)
    return output


def remap_sprite_pixels(sprite_data: bytearray):
    replace_end_markers(sprite_data)
    remap_passenger_pixels(sprite_data, sprite_man_adr, sprite_man_remap)
    remap_passenger_pixels(sprite_data, sprite_woman_adr, sprite_woman_remap)

    for i in range(len(sprite_data)):
        pix = check_transparency(i)
        p0 = sprite_data[i] & 0xf
        p1 = (sprite_data[i] >> 4) & 0xf
        if p0 == 0xa:
            p0 = pix
        if p1 == 0xa:
            p1 = pix
        sprite_data[i] = ((p1 << 4) | p0)

    return sprite_data


def replace_end_markers(sprite_data: bytearray):
    for i in range(3):
        start = sprite_transparency_map[i][0]*4
        end = sprite_transparency_map[i][1]*4

        for ii in range(start, end, 1):
            p0 = sprite_data[ii] & 0xf
            p1 = (sprite_data[ii] >> 4) & 0xf
            if p0 == 0xf:
                p0 = 1
            if p1 == 0xf:
                p1 = 1
            sprite_data[ii] = (((p1 & 9) << 4) | (p0 & 9))
    return


def check_transparency(entry):
    for i in (range(len(sprite_transparency_map))):
        start = sprite_transparency_map[i][0]*4
        end = sprite_transparency_map[i][1]*4
        if start <= entry <= end:
            return sprite_transparency_map[i][2]
    return 0


def remap_passenger_pixels(sprite_data: bytearray, pass_adr: list, remap_data: list):
    for i in (range(len(pass_adr))):
        start = pass_adr[i][0]*4
        end = pass_adr[i][1]*4
        for ii in range(start, end, 1):
            p0 = sprite_data[ii] & 0xf
            p1 = (sprite_data[ii] >> 4) & 0xf
            sprite_data[ii] = ((remap_data[p1] << 4) | remap_data[p0])
    return

# -------------------------------------------------------------------------------------------------
# AUDIO SAMPLE CONVERSION
# -------------------------------------------------------------------------------------------------

TARGET_HZ = 11025       # Target resample rate

# Vol, Delay
postprocess_rebound = [
        [0x3f, 0x0c],
        [0x2e, 0x50],
        [0x0f, 0x05],
        [0x06, 0x05],
        [0x02, 0x50],
]

postprocess_crash1 = [
        [0x38, 0x03],
        [0x28, 0x05],
        [0x1c, 0x04],
        [0x0a, 0x06],
        [0x04, 0x06],
        [0x02, 0x06],
]
# Not entirely sure where in the original code the delay is multiplied by 4, but it works
postprocess_crash2 = [
        [0x3f, 0x06*4],
        [0x28, 0x05*4],
        [0x18, 0x04*4],
        [0x08, 0x03*4],
        [0x04, 0x06*4],
        [0x02, 0x01*4],
        [0x01, 0x01*4],
]

postprocess_waves = [
        [0x14, 0x49],
        [0x06, 0x49],
]

# Format is: Bank Number, Start Offset in Bank, End Offset in Bank, Source Frequency
#            Pre-Padding (11.025 * delay * 8), Volume Multiplier, Repeat, Post Processing Data
fx_vol:float = 0.33
fx_sample_info = [
        ["sample_0",  0, 0x5600, 0x6700,  9000, 88,   fx_vol, 0],                      # Voice: Get Ready
        ["sample_1",  0, 0x6700, 0x7d00,  9000, 88,   fx_vol, 0],                      # Voice: Checkpoint
        ["sample_2a", 1, 0x4000, 0x8000, 10000, 0,    0.25,   0],                      # Cheers / Crowd Noise @ 10,000Hz
        ["sample_2b", 1, 0x4000, 0x8000,  9625, 0,    0.25,   0],                      # Cheers / Crowd Noise @ 9,625Hz
        ["sample_3",  2, 0x0000, 0x0300,  4010, 0,    fx_vol, 0],                      # Slip
        ["sample_4",  2, 0x0900, 0x2000,  8000, 352,  fx_vol, 0, postprocess_crash1],  # Crash 1 (Loud)
        ["sample_5",  2, 0x2000, 0x3c00, 20000, 352,  fx_vol, 0, postprocess_rebound], # Rebound
        ["sample_6",  2, 0x3c00, 0x5c00, 16000, 264,  fx_vol, 0, postprocess_crash2],  # Crash 2 (Soft)
        ["sample_7",  2, 0x6d0d, 0x7400, 14010, 0,    fx_vol, 0],                      # Safety Zone
        ["sample_8",  3, 0x0000, 0x2600,  9000, 88,   fx_vol, 0],                      # Voice: Congratulations
        ["sample_9",  3, 0x2600, 0x8000, 10000, 1674, fx_vol, 0, postprocess_waves],   # Wave
    ]

# Pre-rendered samples to append to PCM data structure (captured from YM)
fx_append_info = [
        ["ym_sample_10", "in/fm_coin.raw"],                # YM: Coin-Op!
        ["ym_sample_11", "in/fm_start_beep1.raw"],         # YM: Countdown Beep 1
        ["ym_sample_12", "in/fm_start_beep2.raw"],         # YM: Countdown Beep 2
        ["ym_sample_13", "in/fm_checkpoint_beeps.raw"],    # YM: Checkpoint Beeps
]

engine_repeat:int = 0
engine_vol:float = 0.15

# Ferrari Engine Samples - the export order matters here
engine_sample_info = [
        # Start line revs
        ["sample_revs_8000",  0, 0x3600, 0x5600,  8004, 0, 0.3, engine_repeat],
        ["sample_revs_8500",  0, 0x3600, 0x5600,  8502, 0, 0.3, engine_repeat],

        # Low revs
        ["sample_tlow_10000", 0, 0x004a, 0x600, 10000, 0, 0.6, engine_repeat],
        ["sample_tlow_10250", 0, 0x004a, 0x600, 10250, 0, 0.6, engine_repeat],

        ["sample_t_8000_v3", 1, 0x0082, 0x0700, 8010, 0, engine_vol-0.10, engine_repeat],
        ["sample_t_8500_v3", 1, 0x0082, 0x0700, 8500, 0, engine_vol-0.08, engine_repeat],
        ["sample_t_9000_v3", 1, 0x0082, 0x0700, 9000, 0, engine_vol-0.06, engine_repeat],
        ["sample_t_9500_v3", 1, 0x0082, 0x0700, 9504, 0, engine_vol-0.04, engine_repeat],
        ["sample_t_10000_v3", 1, 0x0082, 0x0700, 10006, 0, engine_vol-0.03, engine_repeat],
        ["sample_t_10500_v3", 1, 0x0082, 0x0700, 10510, 0, engine_vol-0.02, engine_repeat],
        ["sample_t_11000_v3", 1, 0x0082, 0x0700, 11000, 0, engine_vol, engine_repeat],
        ["sample_t_11500_v3", 1, 0x0082, 0x0700, 11510, 0, engine_vol, engine_repeat],
        ["sample_t_12000_v3", 1, 0x0082, 0x0700, 12000, 0, engine_vol, engine_repeat],
        ["sample_t_12500_v3", 1, 0x0082, 0x0700, 12500, 0, engine_vol, engine_repeat],
        ["sample_t_13000_v3", 1, 0x0082, 0x0700, 13000, 0, engine_vol, engine_repeat],
        ["sample_t_13500_v3", 1, 0x0082, 0x0700, 13500, 0, engine_vol, engine_repeat],
        ["sample_t_14000_v3", 1, 0x0082, 0x0700, 14000, 0, engine_vol, engine_repeat],
        ["sample_t_14500_v3", 1, 0x0082, 0x0700, 14500, 0, engine_vol, engine_repeat],
        ["sample_t_15000_v3", 1, 0x0082, 0x0700, 15000, 0, engine_vol, engine_repeat],
        ["sample_t_15500_v3", 1, 0x0082, 0x0700, 15500, 0, engine_vol, engine_repeat],
        ["sample_t_16000_v3", 1, 0x0082, 0x0700, 16000, 0, engine_vol, engine_repeat],
        ["sample_t_16500_v3", 1, 0x0082, 0x0700, 16500, 0, engine_vol, engine_repeat],
        ["sample_t_17000_v3", 1, 0x0082, 0x0700, 17000, 0, engine_vol, engine_repeat],
        ["sample_t_17500_v3", 1, 0x0082, 0x0700, 17500, 0, engine_vol, engine_repeat],
        ["sample_t_18000_v3", 1, 0x0082, 0x0700, 18000, 0, engine_vol, engine_repeat],
        ["sample_t_18500_v3", 1, 0x0082, 0x0700, 18500, 0, engine_vol, engine_repeat],
        ["sample_t_19000_v3", 1, 0x0082, 0x0700, 19000, 0, engine_vol, engine_repeat],
        ["sample_t_19500_v3", 1, 0x0082, 0x0700, 19500, 0, engine_vol, engine_repeat],
        ["sample_t_20000_v3", 1, 0x0082, 0x0700, 20000, 0, engine_vol, engine_repeat],
        ["sample_t_20500_v3", 1, 0x0082, 0x0700, 20500, 0, engine_vol, engine_repeat],
        ["sample_t_21000_v3", 1, 0x0082, 0x0700, 21000, 0, engine_vol, engine_repeat],
        ["sample_t_21500_v3", 1, 0x0082, 0x0700, 21500, 0, engine_vol, engine_repeat],
        ["sample_t_22000_v3", 1, 0x0082, 0x0700, 22000, 0, engine_vol, engine_repeat],
        ["sample_t_22500_v3", 1, 0x0082, 0x0700, 22500, 0, engine_vol, engine_repeat],
        ["sample_t_23000_v3", 1, 0x0082, 0x0700, 23000, 0, engine_vol, engine_repeat],
        ["sample_t_23500_v3", 1, 0x0082, 0x0700, 23500, 0, engine_vol, engine_repeat],
        ["sample_t_24000_v3", 1, 0x0082, 0x0700, 24000, 0, engine_vol, engine_repeat],
        ["sample_t_24500_v3", 1, 0x0082, 0x0700, 24500, 0, engine_vol, engine_repeat],
        ["sample_t_25000_v3", 1, 0x0082, 0x0700, 25000, 0, engine_vol, engine_repeat],
        ["sample_t_25500_v3", 1, 0x0082, 0x0700, 25500, 0, engine_vol, engine_repeat],
        ["sample_t_26000_v3", 1, 0x0082, 0x0700, 26000, 0, engine_vol, engine_repeat],
        ["sample_t_26500_v3", 1, 0x0082, 0x0700, 26500, 0, engine_vol, engine_repeat],
        ["sample_t_27000_v3", 1, 0x0082, 0x0700, 27000, 0, engine_vol, engine_repeat],
        ["sample_t_27500_v3", 1, 0x0082, 0x0700, 27500, 0, engine_vol, engine_repeat],
        ["sample_t_28000_v3", 1, 0x0082, 0x0700, 28000, 0, engine_vol, engine_repeat],
        ["sample_t_28500_v3", 1, 0x0082, 0x0700, 28500, 0, engine_vol, engine_repeat],
        ["sample_t_29000_v3", 1, 0x0082, 0x0700, 29000, 0, engine_vol, engine_repeat],
        ["sample_t_29500_v3", 1, 0x0082, 0x0700, 29500, 0, engine_vol, engine_repeat],

        # Traffic Specific Samples @ 5 Volume Levels (Some shared with Ferrari Engine)
        # The export order does not matter here
        ["sample_t_11000_v1", 1, 0x0082, 0x700, 11040, 0, 0.05, engine_repeat],  # 11,000 kHz -1000
        ["sample_t_12000_v1", 1, 0x0082, 0x700, 12030, 0, 0.05, engine_repeat],  # 12,000 kHz
        ["sample_t_12250_v1", 1, 0x0082, 0x700, 12250, 0, 0.05, engine_repeat],  # 12,250 kHz +250
        ["sample_t_12500_v1", 1, 0x0082, 0x700, 12590, 0, 0.05, engine_repeat],  # 12,500 kHz +500
        ["sample_t_15000_v1", 1, 0x0082, 0x700, 15070, 0, 0.05, engine_repeat],  # 15,000 kHz -1000
        ["sample_t_16000_v1", 1, 0x0082, 0x700, 16080, 0, 0.05, engine_repeat],  # 16,000 kHz
        ["sample_t_16250_v1", 1, 0x0082, 0x700, 16250, 0, 0.05, engine_repeat],  # 16,250 kHz +250
        ["sample_t_16500_v1", 1, 0x0082, 0x700, 16540, 0, 0.05, engine_repeat],  # 16,500 kHz +500

        ["sample_t_11000_v2", 1, 0x0082, 0x700, 11040, 0, 0.1, engine_repeat],  # 11,000 kHz -1000
        ["sample_t_12000_v2", 1, 0x0082, 0x700, 12030, 0, 0.1, engine_repeat],  # 12,000 kHz
        ["sample_t_12250_v2", 1, 0x0082, 0x700, 12250, 0, 0.1, engine_repeat],  # 12,250 kHz +250
        ["sample_t_12500_v2", 1, 0x0082, 0x700, 12590, 0, 0.1, engine_repeat],  # 12,500 kHz +500
        ["sample_t_15000_v2", 1, 0x0082, 0x700, 15070, 0, 0.1, engine_repeat],  # 15,000 kHz -1000
        ["sample_t_16000_v2", 1, 0x0082, 0x700, 16080, 0, 0.1, engine_repeat],  # 16,000 kHz
        ["sample_t_16250_v2", 1, 0x0082, 0x700, 16250, 0, 0.1, engine_repeat],  # 16,250 kHz +250
        ["sample_t_16500_v2", 1, 0x0082, 0x700, 16540, 0, 0.1, engine_repeat],  # 16,500 kHz +500

        ["sample_t_12250_v3", 1, 0x0082, 0x700, 12250, 0, 0.15, engine_repeat],  # 12,250 kHz +250
        ["sample_t_16250_v3", 1, 0x0082, 0x700, 16250, 0, 0.15, engine_repeat],  # 16,250 kHz +250

        ["sample_t_11000_v4", 1, 0x0082, 0x700, 11040, 0, 0.2, engine_repeat],  # 11,000 kHz -1000
        ["sample_t_12000_v4", 1, 0x0082, 0x700, 12030, 0, 0.2, engine_repeat],  # 12,000 kHz
        ["sample_t_12250_v4", 1, 0x0082, 0x700, 12250, 0, 0.2, engine_repeat],  # 12,250 kHz +250
        ["sample_t_12500_v4", 1, 0x0082, 0x700, 12590, 0, 0.2, engine_repeat],  # 12,500 kHz +500
        ["sample_t_15000_v4", 1, 0x0082, 0x700, 15070, 0, 0.2, engine_repeat],  # 15,000 kHz -1000
        ["sample_t_16000_v4", 1, 0x0082, 0x700, 16080, 0, 0.2, engine_repeat],  # 16,000 kHz
        ["sample_t_16250_v4", 1, 0x0082, 0x700, 16250, 0, 0.2, engine_repeat],  # 16,250 kHz +250
        ["sample_t_16500_v4", 1, 0x0082, 0x700, 16540, 0, 0.2, engine_repeat],  # 16,500 kHz +500

        ["sample_t_11000_v5", 1, 0x0082, 0x700, 11040, 0, 0.25, engine_repeat], # 11,000 kHz -1000
        ["sample_t_12000_v5", 1, 0x0082, 0x700, 12030, 0, 0.25, engine_repeat], # 12,000 kHz
        ["sample_t_12250_v5", 1, 0x0082, 0x700, 12250, 0, 0.25, engine_repeat], # 12,250 kHz +250
        ["sample_t_12500_v5", 1, 0x0082, 0x700, 12590, 0, 0.25, engine_repeat], # 12,500 kHz +500
        ["sample_t_15000_v5", 1, 0x0082, 0x700, 15070, 0, 0.25, engine_repeat], # 15,000 kHz -1000
        ["sample_t_16000_v5", 1, 0x0082, 0x700, 16080, 0, 0.25, engine_repeat], # 16,000 kHz
        ["sample_t_16250_v5", 1, 0x0082, 0x700, 16250, 0, 0.25, engine_repeat], # 16,250 kHz +250
        ["sample_t_16500_v5", 1, 0x0082, 0x700, 16540, 0, 0.25, engine_repeat], # 16,500 kHz +500
]

# -------------------------------------------------------------------------------------------------
# Amiga Audio Mixer Pre-processing
# Convert Unsigned 8-Bit OutRun PCM sample to signed 8-Bit sample, volume adjusted
# -------------------------------------------------------------------------------------------------
def audiomixer_preprocess(sample, vol_mult:float):
    # Conversion of samples is a simple divide by the number of channels.
    # However, there is one exception: when using 3 voices, the division
    # results in a slightly uneven result where positive values are slightly
    # higher than negative values.
    #
    # To compensate for this, each positive sample value will be reduced by 1
    # in the case of 3 voice conversion
    SIGN:int = 128

    if vol_mult == 0.33:
        for i in range(len(sample)):
            if sample[i] - SIGN > 0:
                sample[i] -= 1
            sample[i] = int((sample[i] - SIGN) * vol_mult) & 0xff # -128 to sign, then quieten by number of voices
    else:
        for i in range(len(sample)):
            sample[i] = int((sample[i] - SIGN) * vol_mult) & 0xff # -128 to sign, then quieten by number of voices
    return


# -------------------------------------------------------------------------------------------------
# Repeat (or Loop) the original sample
# Useful to optimize the looping of exceptionally short samples, at the expense of memory
# -------------------------------------------------------------------------------------------------
def repeat_audio(sample, repeat:int):
    if not repeat:
        return sample

    new_data = bytearray()
    for i in range(repeat):
        new_data += sample
    return new_data


# -------------------------------------------------------------------------------------------------
# Resample audio to a common rate
# https://librosa.org/doc/latest/generated/librosa.resample.html
# -------------------------------------------------------------------------------------------------
def resample_audio(audio_data, original_rate, target_rate):
    # Librosa assumes 32-bit float PCM data, so convert from 8-bit signed -> 32-bit float
    float_data = np.frombuffer(audio_data, dtype=np.int8).astype(np.float32)
    output = librosa.resample(float_data, orig_sr=original_rate, target_sr=target_rate, res_type='soxr_vhq')
    pcm_resampled = output.astype(np.int8)
    return pcm_resampled.tobytes()


# Pad sample to Long for Amiga Audio Engine
def pad_sample(data: bytes, pad: int, verbose: bool = False):
    sample_length = len(data)
    diff = sample_length % pad
    if diff != 0:
        extra = pad - diff
        padding = bytearray(extra)
        working_data = bytes(data) + padding
        if verbose:
            print("sample length: "+hex(sample_length)+" | pad: "+hex(extra))
        return working_data
    return data

# -------------------------------------------------------------------------------------------------
# Post Process The Sample
# -------------------------------------------------------------------------------------------------
def postprocess_sample(src_data: bytearray, cmds: list, source_hz):
    src_signed = np.frombuffer(src_data, dtype=np.int8)
    dst_data = bytearray()

    for cmd in cmds:
        vol = cmd[0]
        delay = cmd[1]

        # Scale Input Volume: 0 - 1.0 (float)
        vol *= 1.587
        vol = 100 if vol > 100 else vol
        vol /= 100

        # Figure out how many samples to process per millisecond
        clock_speed = 400000                                                # Clock speed of original Z80
        hz_fraction: float = (TARGET_HZ / source_hz)                        # Fraction of original Hz vs resampled
        samples_per_ms: float = (clock_speed / source_hz) * hz_fraction     # Establish time needed

        # A 'delay unit' is 8ms when executed on the Z80
        samples_total = int(samples_per_ms * (delay * 8))

        for i in range(samples_total):
            if i < len(src_data):
                vol_adjust:float = src_signed[i] * vol
                dst_data = dst_data + np.int8(round(vol_adjust)).tobytes()
            else:
                dst_data.append(0)

    return dst_data


# -------------------------------------------------------------------------------------------------
# Perform Processing on each individual sample to prepare it for in-game use.
# The idea here is to minimize run-time calculations when mixing samples.
# -------------------------------------------------------------------------------------------------
def process_samples(sample_data: bytearray, bank_length, sample_info:list, output_structure=False):
    output = bytearray()
    output_offset:int = 0           # Start Offset of processed sample
    output_length:int = 0           # Length of processed sample
    counter:int = 0

    for sample in sample_info:
        # Extract Sample Properties from our global sample structure
        src_offset:int = sample[1] * bank_length
        start:int = sample[2] + src_offset
        end:int = sample[3] + src_offset
        src_rate:int = sample[4]
        pre_pad_bytes:int = sample[5]
        vol_mult:float = sample[6]
        repeat:int = sample[7]
        cmds:list = sample[8] if len(sample) > 8 else []

        # Add silence to the start of certain samples, so we have flexibility
        # in terms of triggering the sample with a delay, without complex code
        if pre_pad_bytes > 0:
            pre_padding = bytearray(pre_pad_bytes)
            output += pre_padding

        # 1/ Extract Original Sample Data from PCM
        extracted = sample_data[start:end]

        # 2/ Convert Unsigned 8-Bit OutRun PCM sample to signed 8-Bit sample, volume adjusted
        audiomixer_preprocess(extracted, vol_mult)

        # 3/ Repeat sample if necessary (useful optimization for very short samples!)
        if repeat > 1: extracted = repeat_audio(extracted, repeat)

        # 4/ Resample extracted audio to common Hz Frequency
        resampled = resample_audio(extracted, src_rate, TARGET_HZ)

        # 5/ Check for alterations to sample (would have been originally performed by Z80)
        if len(cmds):
            post_sample = postprocess_sample(bytearray(resampled), cmds, src_rate)
        else:
            post_sample = bytearray(resampled) # Same as before

        # 6/ Ensure we align to a 4-byte boundary for Amiga AudioMixer
        padded_sample = pad_sample(post_sample, 4, verbose=False)
        output += padded_sample

        output_offset += (output_length + pre_pad_bytes)
        output_length = len(padded_sample)

        # Create ASM friendly structure
        if output_structure:
            print(sample[0]+":")
            print("    dc.l    $" + f'{output_offset:08X}' + "    ; Sample Offset")
            print("    dc.l    $" + f'{output_length:08X}' + "    ; Sample Length", flush=True)

        counter += 1

    return output


# -------------------------------------------------------------------------------------------------
# Append pre-generated samples
# -------------------------------------------------------------------------------------------------
def append_samples(input_data: bytearray, sample_info:list, output_structure=False):
    output = bytearray(input_data)
    output_offset:int = len(input_data)           # Start Offset of processed sample
    output_length:int = 0                         # Length of processed sample

    for sample in sample_info:
        filename = sample[1]
        file_data = read_file(filename)
        padded_sample = pad_sample(file_data, 4, verbose=False)
        output += padded_sample

        output_offset += output_length
        output_length = len(padded_sample)

        if output_structure:
            print(sample[0]+":")
            print("    dc.l    $" + f'{output_offset:08X}' + "    ; Sample Offset")
            print("    dc.l    $" + f'{output_length:08X}' + "    ; Sample Length", flush=True)

    return output

def load_pcm():
    INTERLEAVE:int = 1
    BANK_LENGTH:int = 32768
    sample_data = bytearray(BANK_LENGTH * 4)

    load_rom("in/opr-10193.66", sample_data, BANK_LENGTH * 0, BANK_LENGTH, 0xbcd10dde, INTERLEAVE)
    load_rom("in/opr-10192.67", sample_data, BANK_LENGTH * 1, BANK_LENGTH, 0x770f1270, INTERLEAVE)
    load_rom("in/opr-10191.68", sample_data, BANK_LENGTH * 2, BANK_LENGTH, 0x20a284ab, INTERLEAVE)
    load_rom("in/opr-10190.69", sample_data, BANK_LENGTH * 3, BANK_LENGTH, 0x7cab70e2, INTERLEAVE)
    return sample_data


# -------------------------------------------------------------------------------------------------
# PROGRAM ENTRY POINT
# -------------------------------------------------------------------------------------------------

def main():
    print("... OutRun: Amiga Edition ROM Converter v0.4 --- ReaSSeMBLeR 2025 ...")

    os.makedirs("audio", exist_ok=True)
    os.makedirs("gfx", exist_ok=True)

    # Road Data
    print("    * Attempting to convert Road ROMs...      ", end="")
    road_data = load_road()
    road1_converted = convert_road1(road_data)
    save_to_file(road1_converted, "gfx/road1.bin")
    road2_converted = convert_road2(road1_converted)
    save_to_file(road2_converted, "gfx/road2.bin")
    print("Success")

    # Sprite Data
    print("    * Attempting to convert Sprite ROMs...    ", end="")
    sprite_data = load_sprites()
    sprites_converted = convert_sprites(sprite_data)
    sprites_converted = remap_sprite_pixels(sprites_converted)
    save_to_file(sprites_converted, "gfx/sprites.bin")
    print("Success")

    # Tile Data
    print("    * Attempting to convert Tile ROMs...      ", end="")
    tile_data = load_tiles()
    tiles_converted = convert_tiles(tile_data)
    tiles_converted = remap_tile_pixels(tiles_converted)
    save_to_file(tiles_converted, "gfx/tiles.bin")
    print("Success")

    # PCM Data
    print("    * Attempting to convert PCM Audio ROMs... ", end="")
    BANK_LENGTH:int = 32768
    sample_data = load_pcm()
    pcm_data = process_samples(sample_data, BANK_LENGTH, fx_sample_info, False)
    pcm_data = append_samples(pcm_data, fx_append_info, False)
    if len(pcm_data) > 0:
        save_to_file(pcm_data, "audio/pcm.bin")
    else:
        print("Warning: No audio data written!")
        exit()
    print("Success")

    print("    * Attempting to build PCM Engine Data...  ", end="")
    pcm_engine_data = process_samples(sample_data, BANK_LENGTH, engine_sample_info, False)
    if len(pcm_engine_data) > 0:
        save_to_file(pcm_engine_data, "audio/pcm_engine.bin")
    else:
        print("Warning: No audio engine data written!")
        exit()
    print("Success")

    print("All done. Please find your files in the 'gfx' and 'audio' subdirectories.")
    return


main()
